/*
 * Imports.cpp
 *
 *  Created on: Nov 19, 2013
 *      Author: nino
 */

#include <Imports.h>


void init_infected_structure(input * input_args, infected_structure * p_infected_parameters){
	init_infected_structure(input_args->input_file_name, input_args->file_type, p_infected_parameters);
}

/**
 * Function that parse input arguments and save it to an input struct
 * @param argc
 * @param argv
 * @param input_args struct of input arguments
 */
void read_input_arguments(int argc, char **argv, input * input_args) {

    // default values
    input_args->no_of_iteration_per_cycle = 50;
    input_args->no_of_cycles = 5;
    input_args->epsilon = 0.01;
    input_args->step = 0.01;
    input_args->print_recovered = FALSE;
    input_args->using_giant = FALSE;

    char *nodes = NULL;
    char *p_parameters = NULL;
    char *q_parameters = NULL;
    char *T_parameters = NULL;

    int opt = 0;
    while ((opt = getopt(argc, argv, optString)) != -1) {
        switch (opt) {

            case 'i':
                input_args->input_file_name = (char *) malloc(strlen(optarg) + 1);
                strcpy(input_args->input_file_name, optarg);
                // only for testing
                // strcpy(network_name, input_args->input_file_name);
                break;
            case 't':
                input_args->file_type = optarg;
                if (strcmp(input_args->file_type, "gml") && strcmp(input_args->file_type, "edgelist") && strcmp(input_args->file_type, "weighted") && strcmp(input_args->file_type, "temporal")) {
                    printf("wrong input file type!\n");
                    display_usage();
                    abort();
                }
                break;
            case 'n':
                input_args->no_of_iteration_per_cycle = atoi(optarg);
                break;
            case 'c':
                input_args->no_of_cycles = atoi(optarg);
                break;
            case 'g':
                input_args->using_giant = TRUE;
                break;
            case 'e':
                input_args->epsilon = atof(optarg);
                break;
            case 'v':
                nodes = (char *) malloc(strlen(optarg) + 1);
                strcpy(nodes, optarg);
                break;
            case 'p':
                p_parameters = (char *) malloc(strlen(optarg) + 1);
                strcpy(p_parameters, optarg);
                break;
            case 'q':
                q_parameters = (char *) malloc(strlen(optarg) + 1);
                strcpy(q_parameters, optarg);
                break;
			case 'T':
                //input_args->temporal_threshold = atoi(optarg);
				T_parameters = (char *) malloc(strlen(optarg) + 1);
				strcpy(T_parameters, optarg);
                break;
            case 's':
                input_args->step = atof(optarg);
                break;
            case 'r':
                input_args->print_recovered = TRUE;
                break;
            case 'h': /* fall-through is intentional */
            case '?':
                display_usage();
                abort();

            default:
                /* You won't actually get here. */
                //display_usage();
                abort();
        }
    }

    // checking arguments

    if (input_args->input_file_name == NULL || input_args->no_of_iteration_per_cycle < 1) display_usage();
    if (optind < argc) {
        printf("wrong input argument %s\n", argv[optind]);
        //display_usage();

    }

    //  vector initalisation, p and q values have to exist
    igraph_vector_init(&input_args->nodes, 0);
    if (nodes) {
        parse_group_of_arguments(nodes, 1, "int", &input_args->nodes);
        free(nodes);
    }

    igraph_vector_init(&input_args->p_parameters, 0);
    if (p_parameters) {
        parse_group_of_arguments(p_parameters, input_args->step, "double", &input_args->p_parameters);
        free(p_parameters);
    } else {
        printf("p value/values is missing!\n");
        display_usage();
    }

    igraph_vector_init(&input_args->q_parameters, 0);
    if (q_parameters) {
        parse_group_of_arguments(q_parameters, input_args->step, "double", &input_args->q_parameters);
        free(q_parameters);
    } else {
        printf("q value/values is missing!\n");
    }

    igraph_vector_init(&input_args->T_parameters, 0);
	if (T_parameters) {
		parse_group_of_arguments(T_parameters, 1, "int", &input_args->T_parameters);
		free(T_parameters);
		input_args->temporal_threshold = VECTOR( input_args->T_parameters )[0];
	} else {
		printf("T value/values is missing!\n");
	}

}

void display_usage(void) {
    puts("epidemic - simulate epidemic for giant component of input graph and defined range of nodes");
    puts("Usage: ");
    puts("epidemic filename -i input file name -t file type -n number of iteration per cycle -c number of cycles -v list of nodes");
    puts("-e epsilon -p values -q values -s step for p q values -r print probability of infection for a starting node -g use giant components only\n");
    puts("# input file name, p and q are mandatory");
    puts("# 0 < vertices < no_of_vertices in giant component");
    puts("Default options:");
    puts("- number of iterations per simulation\"200\"");
    puts("- number of simulation cycles \"5\"");
    puts("- epsilon value for checking of result convergence \"0.01\"");
    puts("- all vertices");
    puts("- step \"0.1\"");
    puts("- r 0(FALSE)");

    /* ... */
    exit(EXIT_FAILURE);
}

/**
 * Function for destroying igraph_vectors into an input structure
 * @param input_args
 */
void clear_input_arguments(input * input_args) {
    igraph_vector_destroy(&input_args->nodes);
    igraph_vector_destroy(&input_args->p_parameters);
    igraph_vector_destroy(&input_args->q_parameters);
    igraph_vector_destroy(&input_args->T_parameters);
    free(input_args->input_file_name);
}


void parse_group_of_arguments(char *optarg, double step, const char *type, igraph_vector_t *values) {

    BOOLEAN is_integer;

    // checking input type
    if (strcmp(type, "int") == 0) {
        is_integer = TRUE;
    } else if (strcmp(type, "double") == 0) {
        is_integer = FALSE;
    } else {
        printf("Wrong inserted type in function %s\n", type);
        exit(-1);
    }

    char *position_start = optarg;
    char *position_end;
    char *position;
    int substring_length;
    igraph_real_t range_start, range_end, value;


 //   printf("%s\n", optarg);
    do {

        BOOLEAN is_range = FALSE;

        // extracting start and end of the record (comma is separator)

        position_end = strchr(position_start, ',');
        if (position_end == NULL) {
            substring_length = strlen(position_start);
        } else {
            substring_length = (int) (position_end - position_start);
        }
        // checking if the record is a range of numbers
        for (position = position_start; position < position_start + substring_length; position++) {
            if (*position == '-') {
                is_range = TRUE;
                break;
            }
        }

        // etxtracting records
        // range of numbers
        if (is_range) {
            // finding start and end values and traslation to a range with step 1

            range_start = string_to_translated_range(position_start, is_integer, step);
            range_end = string_to_translated_range(position + 1, is_integer, step);

//            printf("%d %d ", (int) range_start, (int) range_end);
            if (range_end <= range_start) {
                printf("Wrong range value inserted\n");
                exit(-1);
            }

            // taking all numbers from the translated range, returning them to genuine values and
            // inserting them to the output vector
            for (value = range_start; value <= range_end; value++) {
                igraph_vector_push_back(values, value * step);
            }
            // single number
        } else {
            if (*position_start == 0) break; // end of string

            // translate the value to a range with step 1. taking care about type
            value = string_to_translated_range(position_start, is_integer, step);
     //       printf("%d\n", (int) value);
            // return value to genuine value and inserting to the output vector
            igraph_vector_push_back(values, value * step);
        }

        position_start = position_end;
    } while (position_start++);

}

/**
 * Function that take value from a string depending on number type, and translation
 * to a range depending on the step.
 * @param input_string
 * @param is_integer (1 -int, 0 - double)
 * @param step - double value of step for a value from input range
 * @return  - a translated integer value
 */
int string_to_translated_range(char* input_string, BOOLEAN is_integer, double step) {
    int output_value;
    if (is_integer) {
        output_value = atoi(input_string) / step;
    } else {
        output_value = floor(atof(input_string) / step + 0.5);
    }
    return output_value;
}

void read_benchmark_realization(igraph_vector_t * realization, int realization_id, char * filePath, int * source_id){

	printf("Reading realization ...\n");

	char file_sufix[50];
	sprintf(file_sufix, "realization_num_%d.txt", realization_id );
	char fileName[100];
	strcpy(fileName, filePath);
	strcat(fileName, file_sufix);

	char mystring [100];
	// skip first 3 lines
	FILE *fp;
	fp = fopen( fileName,"r");

	char chr;
	fscanf(fp,"%c %s %s %d \n",&chr, mystring, mystring, source_id);
	fgets (mystring , 100 , fp);
	fgets (mystring , 100 , fp);

	for(int i=0; i < igraph_vector_size(realization); ++i){
		int tmp_state;
		fscanf(fp,"%d\n", &tmp_state);

		VECTOR( *realization )[i] = tmp_state;
	}
	fclose(fp);

}

void read_realization_pqT( contagion_param_struct * epidemicParameters , int realization_id, char * filePath){

	char file_sufix[50];
	sprintf(file_sufix, "realization_num_%d.txt", realization_id );
	char fileName[100];
	strcpy(fileName, filePath);
	strcat(fileName, file_sufix);

	FILE *fp;
	fp = fopen( fileName,"r");

	char chr;
	char mystring [100];
	fgets (mystring , 100 , fp);
	//fprintf(fp,"# p: %lf q: %lf T: %d  \n", p, q, T);
	fscanf(fp,"%c %s %lf %s %lf %s %d \n",&chr, mystring, &epidemicParameters->p, mystring, &epidemicParameters->q , mystring, &epidemicParameters->T);

	//printf(" * P = %lf, * q = %lf , * T = %d \n",*p,*q,*T);
	fclose(fp);

}

void load_geometric_distribution(double param, vector<double> *sample){

	FILE *fp;
	char file_name[50];
	int paramInt = (int) round(param*10); //resolution is 1/10
	sprintf(file_name, "/home/nino/ssd/EPI_code/NoiseDistrData/geometric_0_%d.txt", paramInt );
	fp = fopen(file_name,"r");
	int tmp_num;

	while( fscanf(fp,"%d \n", &tmp_num ) == 1 ){
		sample->push_back((double) tmp_num);
	}
	fclose(fp);
}

void load_B_distribution(int alpha, int betta, vector<double> * betta_distr_sample){

	FILE *fp;
	char file_name[50];
	sprintf(file_name, "/home/nino/ssd/EPI_code/NoiseDistrData/betta_distr_%d_%d.txt", alpha, betta );
	fp = fopen(file_name,"r");
	float tmp_num;

	while( fscanf(fp,"%f \n", &tmp_num ) == 1 ){
		betta_distr_sample->push_back((double) tmp_num);
	}
	fclose(fp);
}

void set_epidemic_parameters(input * input_arg, contagion_param_struct * cont_param){
	cont_param->T_value_min = input_arg->temporal_threshold;
	cont_param->T = cont_param->T_value_min;
	cont_param->p = VECTOR(input_arg->p_parameters)[0];
	cont_param->q = VECTOR(input_arg->q_parameters)[0];
}

void load_stand_Norm_distribution(vector<double> *norm_distr_sample){

	FILE *fp;
	char file_name[50];
	sprintf(file_name, "/home/nino/ssd/EPI_code/NoiseDistrData/stand_norm_distr.txt");
	fp = fopen(file_name,"r");
	float tmp_num;

	while( fscanf(fp,"%f \n", &tmp_num ) == 1 ){
		norm_distr_sample->push_back((double) tmp_num);
	}
	fclose(fp);
}

void multiply_distr_sigma(vector<double> * sample, double sigma){
	for(int i = 0 ; i < sample->size(); ++i){
		sample->at(i) = sample->at(i) * sigma;
	}
}
